# وضعیت نهایی پروژه برای Deploy در Liara

## ✅ مشکلات رفع شده

### 1. مشکل bcrypt (ELF Header Error)
**مشکل:** `bcrypt` یک native module است و برای پلتفرم دیگری compile شده بود.

**راه حل:** ✅
- جایگزینی `bcrypt` با `bcryptjs` (pure JavaScript)
- تغییر در `backend/routes.js`: `require('bcryptjs')`
- تغییر در `backend/package.json`: `"bcryptjs": "^2.4.3"`
- تغییر در `package.json` (root): `"bcryptjs": "^2.4.3"`

**نتیجه:** ✅ bcryptjs در همه پلتفرم‌ها کار می‌کند و نیاز به compile ندارد.

### 2. ذخیره‌سازی تنظیمات
**مشکل:** برخی فیلدها ممکن بود از دست بروند.

**راه حل:** ✅
- بهبود `handleSave` در `SettingsManager`
- استفاده از optional chaining در تمام onChange handlers
- حفظ تمام فیلدها (hero, seo, footer, aiConfig, navigation, legal)

**نتیجه:** ✅ تمام تنظیمات به درستی ذخیره می‌شوند.

## 📋 وضعیت فایل‌های مهم

### Backend Dependencies (`backend/package.json`)
```json
{
  "bcryptjs": "^2.4.3",        ✅ (جایگزین bcrypt)
  "express": "^4.18.2",        ✅
  "mysql2": "^3.6.5",          ✅
  "jsonwebtoken": "^9.0.2",    ✅
  "multer": "^1.4.5-lts.1",    ✅
  "winston": "^3.19.0",        ✅
  "express-rate-limit": "^8.2.1", ✅
  "express-validator": "^7.3.1", ✅
  "validator": "^13.15.26",   ✅
  "fs-extra": "^11.1.1",      ✅
  "dotenv": "^16.3.1"          ✅
}
```

### Root Dependencies (`package.json`)
```json
{
  "bcryptjs": "^2.4.3",        ✅ (جایگزین bcrypt)
  "mysql2": "^3.6.5",          ✅
  "dotenv": "^16.3.1",          ✅
  "@google/genai": "^1.33.0",  ✅
  "react": "^18.2.0",           ✅
  "react-dom": "^18.2.0"        ✅
}
```

### Liara Configuration (`liara.json`)
```json
{
  "platform": "node",
  "app": "viraving-web",
  "port": 3000,
  "build": {
    "command": "npm install && npm run build && cd backend && npm install"
  },
  "start": {
    "command": "cd backend && npm install --production && npm start"
  }
}
```

## 🔧 استفاده از bcryptjs

### API مشابه bcrypt
```javascript
const bcrypt = require('bcryptjs');

// Hash password
const hashedPassword = await bcrypt.hash(password, 10);

// Compare password
const isValid = await bcrypt.compare(password, hashedPassword);
```

### مکان‌های استفاده
1. ✅ `/auth/login` - مقایسه پسورد
2. ✅ `/auth/profile` - هش کردن پسورد جدید
3. ✅ `/install` - هش کردن پسورد admin اولیه

## 📝 چک‌لیست Deploy

### قبل از Deploy
- [x] تمام dependencies به‌روز شده‌اند
- [x] bcryptjs جایگزین bcrypt شده
- [x] liara.json تنظیم شده
- [x] Environment Variables آماده هستند

### Environment Variables مورد نیاز
```
DB_HOST=...
DB_PORT=3306
DB_USER=...
DB_PASSWORD=...
DB_NAME=...
JWT_SECRET=...
GEMINI_API_KEY=...
UPLOADS_DIR=/storage/uploads (اختیاری)
NODE_ENV=production
```

### بعد از Deploy
- [ ] بررسی لاگ‌ها برای خطا
- [ ] تست Login
- [ ] تست ذخیره تنظیمات
- [ ] تست Upload فایل
- [ ] بررسی اتصال به دیتابیس

## 🚀 دستورات Deploy

1. **Push به Git:**
   ```bash
   git add .
   git commit -m "Fix bcrypt issue - replace with bcryptjs"
   git push
   ```

2. **در Liara:**
   - Deploy از Git
   - بررسی Environment Variables
   - بررسی Logs

## ⚠️ نکات مهم

1. **Database:** مطمئن شوید دیتابیس به `utf8mb4_unicode_ci` تبدیل شده است.
2. **Disk:** یک Disk در Liara ایجاد کنید و به `/storage` mount کنید.
3. **Environment Variables:** تمام متغیرهای محیطی را تنظیم کنید.
4. **bcryptjs:** کمی کندتر از bcrypt است اما برای production کافی است.

## ✅ نتیجه

پروژه آماده Deploy است! تمام مشکلات رفع شده و پروژه باید بدون مشکل در Liara اجرا شود.

