# اصلاح Rate Limiting برای Settings و Install Endpoints

## ❌ مشکل

خطای 429 (Too Many Requests) برای:
- `/api/settings` (GET و PUT)
- `/api/install/status` (GET)
- `/api/install` (POST)

## 🔍 علت

Rate limiting عمومی (`apiLimiter`) خیلی سخت بود:
- 100 request در 15 دقیقه
- برای endpoint های settings و install که ممکن است چندین بار فراخوانی شوند (مثلاً در حین نصب یا تنظیمات)، این محدودیت خیلی کم است

## ✅ راه‌حل

### 1. افزایش Rate Limit عمومی
- از 100 به 200 request در 15 دقیقه افزایش یافت

### 2. ایجاد Admin Limiter
- یک `adminLimiter` جدید با 500 request در 15 دقیقه برای endpoint های admin

### 3. اعمال Admin Limiter به Settings و Install
- `/api/settings` (GET و PUT)
- `/api/install/status` (GET)
- `/api/install` (POST)

## 🔧 تغییرات انجام شده

### backend/middleware.js

1. افزایش `apiLimiter`:
```javascript
const apiLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 200, // افزایش از 100 به 200
  ...
});
```

2. اضافه کردن `adminLimiter`:
```javascript
const adminLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 500, // 500 request در 15 دقیقه (خیلی ملایم برای admin)
  ...
});
```

3. Export کردن `adminLimiter`:
```javascript
module.exports = {
  apiLimiter,
  adminLimiter, // اضافه شد
  ...
};
```

### backend/routes.js

1. تغییر rate limiting عمومی:
```javascript
// قبل: router.use(apiLimiter);
// بعد:
router.use((req, res, next) => {
  // Skip rate limiting for settings and install endpoints
  if (req.path === '/settings' || req.path.startsWith('/install')) {
    return next();
  }
  // Apply general rate limiting to all other routes
  apiLimiter(req, res, next);
});
```

2. اعمال `adminLimiter` به endpoint های خاص:
```javascript
router.get('/settings', adminLimiter, async (req, res) => { ... });
router.put('/settings', adminLimiter, authenticateToken, sanitizeBody, async (req, res) => { ... });
router.get('/install/status', adminLimiter, async (req, res) => { ... });
router.post('/install', adminLimiter, async (req, res) => { ... });
```

## ✅ نتیجه

- ✅ Rate limiting عمومی: 200 request در 15 دقیقه (افزایش یافت)
- ✅ Rate limiting برای Settings و Install: 500 request در 15 دقیقه (خیلی ملایم)
- ✅ دیگر خطای 429 برای endpoint های admin دریافت نمی‌شود

## 📝 نکته

اگر هنوز خطای 429 دریافت می‌کنید:
1. چند دقیقه صبر کنید تا window reset شود
2. یا می‌توانید `max` را در `adminLimiter` بیشتر کنید (مثلاً 1000)

## ✅ وضعیت: برطرف شده

Rate limiting برای endpoint های admin ملایم‌تر شد.

