const mysql = require('mysql2/promise');
require('dotenv').config();
const logger = require('./logger');

// Database connection pool
let pool = null;

// Initialize database connection
async function initDB() {
  try {
    const config = {
      host: process.env.DB_HOST || 'localhost',
      port: parseInt(process.env.DB_PORT || '3306'),
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'viraving_db',
      waitForConnections: true,
      connectionLimit: 10,
      queueLimit: 0,
      enableKeepAlive: true,
      keepAliveInitialDelay: 0
    };

    pool = mysql.createPool(config);
    
    // Test connection
    const connection = await pool.getConnection();
    logger.info('Database connected successfully');
    connection.release();
    
    // Create tables if they don't exist
    await createTables();
    
    return pool;
  } catch (error) {
    logger.error('Database connection error', { error: error.message, stack: error.stack });
    throw error;
  }
}

// Create all necessary tables
async function createTables() {
  try {
    // Users table
    await pool.execute(`
      CREATE TABLE IF NOT EXISTS users (
        id INT AUTO_INCREMENT PRIMARY KEY,
        username VARCHAR(255) UNIQUE NOT NULL,
        password VARCHAR(255) NOT NULL,
        email VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    // Settings table (stored as JSON)
    await pool.execute(`
      CREATE TABLE IF NOT EXISTS settings (
        id INT AUTO_INCREMENT PRIMARY KEY,
        data JSON NOT NULL,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    // Projects table
    await pool.execute(`
      CREATE TABLE IF NOT EXISTS projects (
        id VARCHAR(255) PRIMARY KEY,
        title VARCHAR(500) NOT NULL,
        category VARCHAR(255),
        image TEXT,
        description TEXT,
        tags JSON,
        client VARCHAR(255),
        year VARCHAR(50),
        projectUrl TEXT,
        seo JSON,
        challenges JSON,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    // Ensure projectUrl column exists on existing databases
    try {
      const [columns] = await pool.execute(`
        SELECT COLUMN_NAME 
        FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_SCHEMA = DATABASE() 
        AND TABLE_NAME = 'projects' 
        AND COLUMN_NAME = 'projectUrl'
      `);
      
      if (columns.length === 0) {
        await pool.execute(`
          ALTER TABLE projects 
          ADD COLUMN projectUrl TEXT
        `);
        logger.info('Added projectUrl column to projects table');
      }
    } catch (error) {
      // Column might already exist or table doesn't exist yet (will be created above)
      logger.warn('Could not add projectUrl column (may already exist)', { error: error.message });
    }

    // Blog posts table
    await pool.execute(`
      CREATE TABLE IF NOT EXISTS blog (
        id VARCHAR(255) PRIMARY KEY,
        title VARCHAR(500) NOT NULL,
        excerpt TEXT,
        content LONGTEXT,
        date VARCHAR(50),
        readTime VARCHAR(50),
        image TEXT,
        author VARCHAR(255),
        seo JSON,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    // Services table
    await pool.execute(`
      CREATE TABLE IF NOT EXISTS services (
        id VARCHAR(255) PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        description TEXT,
        iconName VARCHAR(100),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    // Process steps table
    await pool.execute(`
      CREATE TABLE IF NOT EXISTS process (
        id VARCHAR(255) PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        description TEXT,
        iconName VARCHAR(100),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    // Team members table
    await pool.execute(`
      CREATE TABLE IF NOT EXISTS team (
        id VARCHAR(255) PRIMARY KEY,
        name VARCHAR(255) NOT NULL,
        role VARCHAR(255),
        image TEXT,
        bio TEXT,
        socials JSON,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    // Reviews table
    await pool.execute(`
      CREATE TABLE IF NOT EXISTS reviews (
        id VARCHAR(255) PRIMARY KEY,
        name VARCHAR(255) NOT NULL,
        role VARCHAR(255),
        text TEXT,
        avatar TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    // Messages table
    await pool.execute(`
      CREATE TABLE IF NOT EXISTS messages (
        id VARCHAR(255) PRIMARY KEY,
        name VARCHAR(255) NOT NULL,
        phone VARCHAR(50),
        services JSON,
        description TEXT,
        date VARCHAR(50),
        isRead BOOLEAN DEFAULT FALSE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    logger.info('Database tables created/verified');
  } catch (error) {
    logger.error('Error creating tables', { error: error.message, stack: error.stack });
    throw error;
  }
}

// Get database connection pool
function getPool() {
  if (!pool) {
    throw new Error('Database not initialized. Call initDB() first.');
  }
  return pool;
}

// Initialize database with custom config (for installer)
async function initDBWithConfig(config) {
  try {
    const dbConfig = {
      host: config.host,
      port: parseInt(config.port || '3306'),
      user: config.user,
      password: config.pass,
      database: config.name,
      waitForConnections: true,
      connectionLimit: 10,
      queueLimit: 0
    };

    // Create connection without database first to create it if needed
    const tempPool = mysql.createPool({
      ...dbConfig,
      database: undefined
    });

    // Create database if it doesn't exist
    await tempPool.execute(`CREATE DATABASE IF NOT EXISTS \`${config.name}\``);
    await tempPool.end();

    // Now create pool with database
    pool = mysql.createPool(dbConfig);
    
    // Test connection
    const connection = await pool.getConnection();
    logger.info('Database connected with custom config');
    connection.release();
    
    // Create tables
    await createTables();
    
    return pool;
  } catch (error) {
    logger.error('Database initialization error', { error: error.message, stack: error.stack });
    throw error;
  }
}

module.exports = {
  initDB,
  initDBWithConfig,
  getPool,
  createTables
};
