const rateLimit = require('express-rate-limit');
const { body, validationResult } = require('express-validator');
const validator = require('validator');
const logger = require('./logger');

// --- RATE LIMITING ---

// General API rate limiter (200 requests per 15 minutes - increased for admin panel)
const apiLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 200, // limit each IP to 200 requests per windowMs (increased from 100)
  message: 'Too many requests from this IP, please try again later.',
  standardHeaders: true,
  legacyHeaders: false,
  handler: (req, res) => {
    logger.warn(`Rate limit exceeded for IP: ${req.ip}`);
    res.status(429).json({ 
      message: 'Too many requests from this IP, please try again later.' 
    });
  }
});

// Lenient rate limiter for settings and install endpoints (admin operations)
const adminLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 500, // limit each IP to 500 requests per windowMs (very lenient for admin)
  message: 'Too many requests from this IP, please try again later.',
  standardHeaders: true,
  legacyHeaders: false,
  handler: (req, res) => {
    logger.warn(`Admin rate limit exceeded for IP: ${req.ip}`);
    res.status(429).json({ 
      message: 'Too many requests from this IP, please try again later.' 
    });
  }
});

// Strict rate limiter for authentication endpoints (5 requests per 15 minutes)
const authLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 5, // limit each IP to 5 requests per windowMs
  message: 'Too many login attempts, please try again later.',
  standardHeaders: true,
  legacyHeaders: false,
  skipSuccessfulRequests: true, // Don't count successful requests
  handler: (req, res) => {
    logger.warn(`Auth rate limit exceeded for IP: ${req.ip}`);
    res.status(429).json({ 
      message: 'Too many login attempts, please try again later.' 
    });
  }
});

// File upload rate limiter (10 uploads per hour)
const uploadLimiter = rateLimit({
  windowMs: 60 * 60 * 1000, // 1 hour
  max: 10, // limit each IP to 10 uploads per hour
  message: 'Too many file uploads, please try again later.',
  standardHeaders: true,
  legacyHeaders: false,
  handler: (req, res) => {
    logger.warn(`Upload rate limit exceeded for IP: ${req.ip}`);
    res.status(429).json({ 
      message: 'Too many file uploads, please try again later.' 
    });
  }
});

// --- INPUT VALIDATION & SANITIZATION ---

// Sanitize string input
const sanitizeString = (str) => {
  if (typeof str !== 'string') return str;
  // Remove potentially dangerous characters
  return validator.escape(str.trim());
};

// Sanitize HTML content (for rich text editors)
const sanitizeHTML = (html) => {
  if (typeof html !== 'string') return html;
  // Basic sanitization - remove script tags and dangerous attributes
  return html
    .replace(/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/gi, '')
    .replace(/on\w+="[^"]*"/gi, '')
    .replace(/on\w+='[^']*'/gi, '')
    .replace(/javascript:/gi, '');
};

// Validate email
const validateEmail = (email) => {
  return validator.isEmail(email);
};

// Validate URL
const validateURL = (url) => {
  return validator.isURL(url, { require_protocol: false });
};

// Fields that should NOT be HTML-escaped (they contain technical values)
const NO_ESCAPE_FIELDS = [
  'modelName', // AI model names like "openai/gpt-5-mini:online"
  'apiKey', // API keys should not be escaped
  'endpoint', // API endpoints
  'baseURL', // Base URLs
  'url', // URLs in general
  'href', // Links
  'src', // Image sources
  'canonicalUrl', // SEO canonical URLs
  'ogUrl', // Open Graph URLs
  'ogImage', // Open Graph image URLs
  'schemaUrl', // Schema.org URLs
  'schemaLogo', // Schema.org logo URLs
  'favicon', // Favicon URLs
  'appleTouchIcon', // Apple touch icon URLs
  'projectUrl', // Project URLs
];

// Sanitize object recursively
const sanitizeObject = (obj) => {
  if (obj === null || obj === undefined) return obj;
  if (typeof obj !== 'object') return sanitizeString(String(obj));
  
  if (Array.isArray(obj)) {
    return obj.map(item => sanitizeObject(item));
  }
  
  const sanitized = {};
  for (const key in obj) {
    if (obj.hasOwnProperty(key)) {
      const value = obj[key];
      if (typeof value === 'string') {
        // Don't escape technical fields that contain special characters
        if (NO_ESCAPE_FIELDS.includes(key)) {
          // Only trim, don't escape
          sanitized[key] = value.trim();
        } else if (value.includes('<') && value.includes('>')) {
          // HTML content
          sanitized[key] = sanitizeHTML(value);
        } else {
          // Regular string - escape for XSS protection
          sanitized[key] = sanitizeString(value);
        }
      } else if (typeof value === 'object') {
        sanitized[key] = sanitizeObject(value);
      } else {
        sanitized[key] = value;
      }
    }
  }
  return sanitized;
};

// Middleware to sanitize request body
const sanitizeBody = (req, res, next) => {
  if (req.body) {
    req.body = sanitizeObject(req.body);
  }
  next();
};

// Middleware to sanitize query parameters
const sanitizeQuery = (req, res, next) => {
  if (req.query) {
    req.query = sanitizeObject(req.query);
  }
  next();
};

// Validation rules for common endpoints
const validateProject = [
  body('title').trim().isLength({ min: 1, max: 500 }).withMessage('Title must be between 1 and 500 characters'),
  body('category').optional().trim().isLength({ max: 255 }),
  body('description').optional().trim(),
  body('client').optional().trim().isLength({ max: 255 }),
  body('year').optional().trim().isLength({ max: 50 }),
  body('projectUrl').optional().trim().custom((value) => {
    if (value && !validateURL(value)) {
      throw new Error('Invalid URL format');
    }
    return true;
  }),
  (req, res, next) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }
    next();
  }
];

const validateBlogPost = [
  body('title').trim().isLength({ min: 1, max: 500 }).withMessage('Title must be between 1 and 500 characters'),
  body('excerpt').optional().trim(),
  body('content').optional().trim(),
  body('author').optional().trim().isLength({ max: 255 }),
  (req, res, next) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }
    next();
  }
];

const validateMessage = [
  body('name').trim().isLength({ min: 1, max: 255 }).withMessage('Name is required and must be less than 255 characters'),
  body('phone').optional().trim().isLength({ max: 50 }),
  body('description').optional().trim(),
  (req, res, next) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }
    next();
  }
];

const validateLogin = [
  body('username').trim().isLength({ min: 1 }).withMessage('Username is required'),
  body('password').trim().isLength({ min: 1 }).withMessage('Password is required'),
  (req, res, next) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }
    next();
  }
];

module.exports = {
  apiLimiter,
  adminLimiter,
  authLimiter,
  uploadLimiter,
  sanitizeBody,
  sanitizeQuery,
  sanitizeString,
  sanitizeHTML,
  sanitizeObject,
  validateEmail,
  validateURL,
  validateProject,
  validateBlogPost,
  validateMessage,
  validateLogin
};

