import React, { useState, useEffect } from 'react';
import { Home, Briefcase, Zap, Phone, User, FileText, Settings, Star, Layers } from 'lucide-react';
import { DataService } from '../services/dataService';
import { AppSettings, MobileNavItem } from '../types';

interface BottomNavProps {
  currentRoute: string;
}

// Icon Mapping
const iconMap: Record<string, React.ReactNode> = {
  'Home': <Home size={22} />,
  'Briefcase': <Briefcase size={22} />,
  'Zap': <Zap size={22} />,
  'Phone': <Phone size={22} />,
  'User': <User size={22} />,
  'FileText': <FileText size={22} />,
  'Settings': <Settings size={22} />,
  'Star': <Star size={22} />,
  'Layers': <Layers size={22} />,
};

const BottomNav: React.FC<BottomNavProps> = ({ currentRoute }) => {
  const [navItems, setNavItems] = useState<MobileNavItem[]>([]);

  useEffect(() => {
    const fetchSettings = async () => {
      const settings = await DataService.getSettings();
      if (settings?.navigation?.mobile) {
        setNavItems(settings.navigation.mobile);
      } else {
        // Fallback default
        setNavItems([
          { id: '1', name: 'خانه', href: '#home', iconName: 'Home' },
          { id: '2', name: 'پروژه‌ها', href: '#portfolio', iconName: 'Briefcase' },
          { id: '3', name: 'مشاور AI', href: '#home', anchor: 'ai-consultant', iconName: 'Zap' },
          { id: '4', name: 'تماس', href: '#contact', iconName: 'Phone' },
        ]);
      }
    };
    fetchSettings();
    window.addEventListener('settings-updated', fetchSettings);
    return () => window.removeEventListener('settings-updated', fetchSettings);
  }, []);

  const getActive = (href: string, anchor?: string) => {
    const target = anchor ? `${href}?scroll=${anchor}` : href;
    
    if (href === '#portfolio') {
      return currentRoute === '#portfolio' || currentRoute.startsWith('#project/');
    }
    if (target === '#home?scroll=ai-consultant') {
      return currentRoute === '#home?scroll=ai-consultant';
    }
    if (href === '#home' && !anchor) {
      return currentRoute === '#home' || 
             (currentRoute.startsWith('#home') && currentRoute !== '#home?scroll=ai-consultant');
    }
    if (href === '#contact') {
      return currentRoute.startsWith('#contact');
    }
    // Generic check
    return currentRoute === target;
  };

  const handleNavClick = (e: React.MouseEvent<HTMLAnchorElement>, target: string) => {
    e.preventDefault();
    window.location.hash = target;
  };

  if (navItems.length === 0) return null;

  return (
    <div className="md:hidden fixed bottom-0 left-0 right-0 bg-white/95 dark:bg-slate-900/95 backdrop-blur-xl border-t border-slate-200 dark:border-slate-800 pb-safe z-50 shadow-[0_-5px_10px_rgba(0,0,0,0.03)] dark:shadow-[0_-5px_10px_rgba(0,0,0,0.2)]">
      <div className="flex justify-around items-center h-16 px-2">
        {navItems.map((item) => {
          const target = item.anchor ? `${item.href}?scroll=${item.anchor}` : item.href;
          const isActive = getActive(item.href, item.anchor);
          
          return (
            <a
              key={item.id}
              href={target}
              onClick={(e) => handleNavClick(e, target)}
              className="flex flex-col items-center justify-center w-full h-full relative cursor-pointer"
            >
              {/* Active Indicator Background (Subtle Pill) */}
              {isActive && (
                <div className="absolute top-1 w-12 h-8 bg-indigo-100 dark:bg-indigo-900/50 rounded-xl -z-10 animate-fade-in"></div>
              )}
              
              <div className={`transition-all duration-300 ${isActive ? 'text-indigo-600 dark:text-indigo-400 scale-105' : 'text-slate-400 dark:text-slate-500 hover:text-slate-600 dark:hover:text-slate-300'}`}>
                {iconMap[item.iconName] || iconMap['Home']}
              </div>
              <span className={`text-[10px] font-medium mt-1 transition-colors ${isActive ? 'text-indigo-600 dark:text-indigo-400' : 'text-slate-400 dark:text-slate-500'}`}>
                {item.name}
              </span>
            </a>
          );
        })}
      </div>
    </div>
  );
};

export default BottomNav;