import React, { useState, useEffect } from 'react';
import { Send, MapPin, Phone, Mail, Loader2, AlertCircle } from 'lucide-react';
import { DataService, fireToast } from '../services/dataService';

const serviceOptions = [
  'طراحی سایت',
  'طراحی اپلیکیشن',
  'سئو و مارکتینگ',
  'طراحی UI/UX',
  'مشاوره اختصاصی'
];

const Contact: React.FC = () => {
  const [selectedServices, setSelectedServices] = useState<string[]>([]);
  const [formData, setFormData] = useState({ name: '', phone: '', description: '' });
  const [errors, setErrors] = useState<{name?: string, phone?: string}>({});
  const [submitted, setSubmitted] = useState(false);
  const [isSending, setIsSending] = useState(false);
  const [contactInfo, setContactInfo] = useState<{ address?: string; phone?: string; email?: string }>({});

  useEffect(() => {
    const loadSettings = async () => {
      try {
        const settings = await DataService.getSettings();
        const footer = settings?.footer;
        setContactInfo({
          address: footer?.contact?.address,
          phone: footer?.contact?.phone,
          email: footer?.contact?.email,
        });
      } catch (e) {
        console.error('Error loading settings for contact info:', e);
      }
    };
    loadSettings();
  }, []);

  const toggleService = (service: string) => {
    if (selectedServices.includes(service)) {
      setSelectedServices(selectedServices.filter(s => s !== service));
    } else {
      setSelectedServices([...selectedServices, service]);
    }
  };

  const validate = () => {
    let isValid = true;
    const newErrors: any = {};

    if (!formData.name.trim()) {
      newErrors.name = 'نام الزامی است';
      isValid = false;
    }

    const phoneRegex = /^09\d{9}$/;
    if (!formData.phone || !phoneRegex.test(formData.phone)) {
      newErrors.phone = 'شماره موبایل نامعتبر است (مثال: 09123456789)';
      isValid = false;
    }

    setErrors(newErrors);
    return isValid;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!validate()) return;

    setIsSending(true);
    try {
      await DataService.addMessage({
        name: formData.name,
        phone: formData.phone,
        description: formData.description,
        services: selectedServices
      });
      setSubmitted(true);
      fireToast('پیام شما با موفقیت ارسال شد', 'success');
      setFormData({ name: '', phone: '', description: '' });
      setSelectedServices([]);
      setErrors({});
      setTimeout(() => setSubmitted(false), 5000);
    } catch (error) {
      fireToast('خطا در ارسال پیام. لطفا مجددا تلاش کنید.', 'error');
    } finally {
      setIsSending(false);
    }
  };

  return (
    <section className="min-h-screen bg-slate-50 dark:bg-slate-950 pt-20 md:pt-32 pb-24 animate-fade-in transition-colors duration-300">
      <div className="container mx-auto px-4">
        
        <div className="text-center mb-12 mt-8">
           <h2 className="text-3xl md:text-5xl font-bold text-slate-900 dark:text-white mb-4">شروع همکاری</h2>
           <p className="text-slate-600 dark:text-slate-400 max-w-xl mx-auto">
             فرم زیر را تکمیل کنید، ما پروژه شما را تحلیل کرده و با شما تماس می‌گیریم.
           </p>
        </div>

        <div className="grid lg:grid-cols-2 gap-8 lg:gap-12 items-start max-w-6xl mx-auto">
          
          {/* Enhanced Order Form */}
          <div className="bg-white dark:bg-slate-900 p-6 md:p-8 rounded-3xl border border-slate-200 dark:border-slate-800 shadow-2xl relative overflow-hidden transition-colors duration-300">
            <div className="absolute top-0 left-0 w-full h-2 bg-gradient-to-r from-indigo-500 via-teal-500 to-slate-500"></div>
            
            {submitted ? (
              <div className="h-96 flex flex-col items-center justify-center text-center animate-fade-in">
                 <div className="w-20 h-20 bg-green-100 dark:bg-green-500/20 text-green-600 dark:text-green-400 rounded-full flex items-center justify-center mb-6">
                   <Send size={36} />
                 </div>
                 <h3 className="text-2xl font-bold text-slate-900 dark:text-white mb-2">درخواست شما ثبت شد!</h3>
                 <p className="text-slate-500 dark:text-slate-400">کارشناسان ما به زودی با شما تماس خواهند گرفت.</p>
                 <button onClick={() => setSubmitted(false)} className="mt-6 text-indigo-600 hover:underline text-sm">ارسال درخواست جدید</button>
              </div>
            ) : (
              <form className="space-y-6" onSubmit={handleSubmit}>
                
                {/* Service Selection */}
                <div>
                  <label className="block text-slate-900 dark:text-white font-medium text-sm mb-3">نوع پروژه (انتخاب چند مورد آزاد است)</label>
                  <div className="flex flex-wrap gap-2">
                    {serviceOptions.map(service => (
                      <button
                        key={service}
                        type="button"
                        onClick={() => toggleService(service)}
                        className={`px-4 py-2 rounded-xl text-sm font-medium transition-all border ${
                          selectedServices.includes(service)
                            ? 'bg-indigo-600 text-white border-indigo-500 shadow-lg shadow-indigo-500/20'
                            : 'bg-slate-50 dark:bg-slate-950 text-slate-600 dark:text-slate-400 border-slate-200 dark:border-slate-800 hover:border-slate-400 dark:hover:border-slate-600'
                        }`}
                      >
                        {service}
                      </button>
                    ))}
                  </div>
                </div>

                <div className="grid md:grid-cols-2 gap-6">
                  <div>
                    <label className="block text-slate-500 dark:text-slate-400 text-sm mb-2">نام کامل <span className="text-red-500">*</span></label>
                    <input type="text" value={formData.name} onChange={e => { setFormData({...formData, name: e.target.value}); if(errors.name) setErrors({...errors, name: undefined}); }} className={`w-full bg-slate-50 dark:bg-slate-950 border rounded-xl px-4 py-3 text-slate-900 dark:text-white focus:outline-none focus:border-indigo-500 transition-colors ${errors.name ? 'border-red-500' : 'border-slate-200 dark:border-slate-800'}`} placeholder="مثلا: علی محمدی" />
                    {errors.name && <p className="text-red-500 text-xs mt-1 flex items-center gap-1"><AlertCircle size={12} /> {errors.name}</p>}
                  </div>
                  <div>
                    <label className="block text-slate-500 dark:text-slate-400 text-sm mb-2">شماره موبایل <span className="text-red-500">*</span></label>
                    <input type="tel" dir="ltr" value={formData.phone} onChange={e => { setFormData({...formData, phone: e.target.value}); if(errors.phone) setErrors({...errors, phone: undefined}); }} className={`w-full bg-slate-50 dark:bg-slate-950 border rounded-xl px-4 py-3 text-slate-900 dark:text-white focus:outline-none focus:border-indigo-500 transition-colors placeholder:text-right ${errors.phone ? 'border-red-500' : 'border-slate-200 dark:border-slate-800'}`} placeholder="09xxxxxxxxx" />
                     {errors.phone && <p className="text-red-500 text-xs mt-1 flex items-center gap-1"><AlertCircle size={12} /> {errors.phone}</p>}
                  </div>
                </div>
                
                <div>
                  <label className="block text-slate-500 dark:text-slate-400 text-sm mb-2">توضیحات تکمیلی</label>
                  <textarea rows={4} value={formData.description} onChange={e => setFormData({...formData, description: e.target.value})} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl px-4 py-3 text-slate-900 dark:text-white focus:outline-none focus:border-indigo-500 transition-colors placeholder-slate-400 dark:placeholder-slate-600" placeholder="درباره ایده یا نیازهای پروژه خود بنویسید..."></textarea>
                </div>

                <button type="submit" disabled={isSending} className="w-full bg-indigo-600 hover:bg-indigo-700 text-white font-bold py-4 rounded-xl transition-all flex items-center justify-center gap-2 shadow-lg shadow-indigo-600/20 hover:shadow-indigo-600/40 transform hover:-translate-y-1 disabled:opacity-70 disabled:cursor-not-allowed">
                  {isSending ? (
                    <>
                      <Loader2 size={18} className="animate-spin" />
                      در حال ارسال...
                    </>
                  ) : (
                    <>
                      ارسال درخواست
                      <Send size={18} />
                    </>
                  )}
                </button>
              </form>
            )}
          </div>

          {/* Info Side */}
          <div className="space-y-6">
             <div className="bg-gradient-to-br from-indigo-100/50 to-white/50 dark:from-indigo-900/50 dark:to-slate-900/50 p-8 rounded-3xl border border-indigo-100 dark:border-slate-800">
                <h3 className="text-xl font-bold text-slate-900 dark:text-white mb-4">چرا ویراوینگ؟</h3>
                <ul className="space-y-3 text-slate-600 dark:text-slate-300 text-sm list-disc list-inside marker:text-indigo-500">
                  <li>مشاوره فنی رایگان پیش از قرارداد</li>
                  <li>ضمانت کیفیت کدنویسی و امنیت</li>
                  <li>پشتیبانی ۶ ماهه رایگان</li>
                  <li>طراحی رابط کاربری اختصاصی (بدون قالب آماده)</li>
                </ul>
             </div>

            <div className="space-y-4">
              <div className="flex items-center gap-4 p-5 bg-white dark:bg-slate-900/50 rounded-2xl border border-slate-200 dark:border-slate-800 shadow-sm">
                <div className="w-10 h-10 bg-slate-100 dark:bg-slate-800 text-indigo-500 dark:text-indigo-400 rounded-lg flex items-center justify-center shrink-0">
                  <MapPin size={20} />
                </div>
                <div>
                  <p className="text-slate-500 text-xs">آدرس ما</p>
                  <p className="text-slate-800 dark:text-slate-300 text-sm font-medium">
                    {contactInfo.address || 'آدرس هنوز ثبت نشده است'}
                  </p>
                </div>
              </div>

              <div className="flex items-center gap-4 p-5 bg-white dark:bg-slate-900/50 rounded-2xl border border-slate-200 dark:border-slate-800 shadow-sm">
                <div className="w-10 h-10 bg-slate-100 dark:bg-slate-800 text-green-500 dark:text-green-400 rounded-lg flex items-center justify-center shrink-0">
                  <Phone size={20} />
                </div>
                <div>
                   <p className="text-slate-500 text-xs">تماس مستقیم</p>
                   <p className="text-slate-800 dark:text-slate-300 text-sm font-medium">
                     {contactInfo.phone || 'شماره تماس هنوز ثبت نشده است'}
                   </p>
                </div>
              </div>

              <div className="flex items-center gap-4 p-5 bg-white dark:bg-slate-900/50 rounded-2xl border border-slate-200 dark:border-slate-800 shadow-sm">
                <div className="w-10 h-10 bg-slate-100 dark:bg-slate-800 text-pink-500 dark:text-pink-400 rounded-lg flex items-center justify-center shrink-0">
                  <Mail size={20} />
                </div>
                <div>
                   <p className="text-slate-500 text-xs">ارسال ایمیل</p>
                   <p className="text-slate-800 dark:text-slate-300 text-sm font-medium">
                     {contactInfo.email || 'ایمیل هنوز ثبت نشده است'}
                   </p>
                </div>
              </div>
            </div>
          </div>

        </div>
      </div>
    </section>
  );
};

export default Contact;