import React, { useEffect, useState } from 'react';
import { ArrowRight, Calendar, User, ExternalLink, Tag, Layers, Loader2 } from 'lucide-react';
import { DataService } from '../services/dataService';
import { Project } from '../types';

interface ProjectDetailProps {
  id: string;
}

const ProjectDetail: React.FC<ProjectDetailProps> = ({ id }) => {
  const [project, setProject] = useState<Project | null>(null);
  const [relatedProjects, setRelatedProjects] = useState<Project[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchProject = async () => {
      setLoading(true);
      const found = await DataService.getProjectById(id);
      setProject(found || null);
      
      if (found) {
          // Fetch related projects based on the same category
          const response = await DataService.getProjects(1, 4, undefined, found.category);
          // Filter out the current project and limit to 3
          const related = response.data.filter(p => p.id !== found.id).slice(0, 3);
          setRelatedProjects(related);
      }
      
      setLoading(false);
      window.scrollTo(0, 0); // Ensure scroll to top
    };
    fetchProject();
  }, [id]);

  const handleNavClick = (e: React.MouseEvent<HTMLAnchorElement>, target: string) => {
    e.preventDefault();
    window.location.hash = target;
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center pt-20">
        <Loader2 className="w-10 h-10 animate-spin text-indigo-600" />
      </div>
    );
  }

  if (!project) {
    return (
      <div className="min-h-screen pt-32 flex flex-col items-center justify-center text-slate-500 dark:text-slate-400">
        <h2 className="text-2xl font-bold mb-4">پروژه یافت نشد</h2>
        <a href="#portfolio" onClick={(e) => handleNavClick(e, '#portfolio')} className="text-indigo-600 hover:underline">بازگشت به نمونه‌کارها</a>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-slate-50 dark:bg-slate-950 pb-24 md:pt-32 animate-fade-in transition-colors duration-300">
      {/* Mobile Header with Back Button - High Z-Index to stay on top */}
      <div className="md:hidden fixed top-0 left-0 right-0 z-[60] p-4 flex items-center gap-3 bg-white/90 dark:bg-slate-900/90 backdrop-blur-lg border-b border-slate-200 dark:border-slate-800 shadow-xl">
        <a 
          href="#portfolio" 
          onClick={(e) => handleNavClick(e, '#portfolio')}
          className="p-2 -mr-2 text-slate-600 dark:text-slate-300 hover:text-slate-900 dark:hover:text-white active:scale-95 transition-transform"
        >
          <ArrowRight size={24} />
        </a>
        <h1 className="font-bold text-slate-900 dark:text-white text-lg truncate flex-1">{project.title}</h1>
      </div>

      {/* Hero Image */}
      <div className="relative w-full h-[50vh] md:h-[60vh] mt-16 md:mt-0">
        <img 
          src={project.image} 
          alt={project.title} 
          className="w-full h-full object-cover"
        />
        <div className="absolute inset-0 bg-gradient-to-t from-slate-900 via-slate-900/40 to-transparent"></div>
        
        {/* Desktop Back Button - Always white due to image background */}
        <div className="hidden md:block absolute top-8 right-8 z-10">
           <a 
             href="#portfolio" 
             onClick={(e) => handleNavClick(e, '#portfolio')}
             className="flex items-center gap-2 bg-slate-900/50 backdrop-blur-md text-white px-5 py-2.5 rounded-xl hover:bg-slate-900 transition-all hover:scale-105 border border-white/10"
            >
             <ArrowRight size={20} />
             بازگشت به پروژه‌ها
           </a>
        </div>

        <div className="absolute bottom-0 right-0 left-0 p-6 md:p-12 container mx-auto">
          <span className="inline-block px-3 py-1 bg-indigo-600 text-white text-xs font-bold rounded-full mb-4 uppercase tracking-wider shadow-lg shadow-indigo-600/30">
            {project.category}
          </span>
          <h1 className="text-3xl md:text-6xl font-black text-white mb-2 shadow-black drop-shadow-lg leading-tight">
            {project.title}
          </h1>
        </div>
      </div>

      <div className="container mx-auto px-4 mt-8 grid lg:grid-cols-3 gap-12">
        {/* Main Content */}
        <div className="lg:col-span-2 space-y-8">
          <div className="bg-white dark:bg-slate-900 rounded-3xl p-6 md:p-8 border border-slate-200 dark:border-slate-800 shadow-xl">
            <h2 className="text-2xl font-bold text-slate-900 dark:text-white mb-4 flex items-center gap-2">
              <Layers className="text-indigo-600 dark:text-indigo-500" />
              درباره پروژه
            </h2>
            <p className="text-slate-600 dark:text-slate-300 leading-8 text-lg text-justify">
              {project.description}
            </p>
          </div>

          {/* Dynamic Challenges Section */}
          {project.challenges && project.challenges.length > 0 && (
            <div className="bg-white dark:bg-slate-900 rounded-3xl p-6 md:p-8 border border-slate-200 dark:border-slate-800 shadow-xl">
              <h3 className="text-xl font-bold text-slate-900 dark:text-white mb-6">چالش‌ها و راه‌حل‌ها</h3>
              <div className="space-y-4">
                {project.challenges.map((challenge, index) => (
                  <div key={challenge.id || index} className="flex gap-4 p-4 rounded-xl bg-slate-50 dark:bg-slate-800/30 border border-slate-100 dark:border-slate-800/50">
                    <div className="w-8 h-8 rounded-full bg-red-100 dark:bg-red-500/20 text-red-600 dark:text-red-400 flex items-center justify-center shrink-0 font-bold border border-red-200 dark:border-red-500/20">
                      {index + 1}
                    </div>
                    <div>
                      <h4 className="font-bold text-slate-800 dark:text-white mb-1">{challenge.title}</h4>
                      <p className="text-slate-700 dark:text-slate-300 text-sm leading-relaxed">{challenge.description}</p>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          )}
        </div>

        {/* Sidebar Info */}
        <div className="space-y-6">
          <div className="bg-white dark:bg-slate-900 rounded-3xl p-6 border border-slate-200 dark:border-slate-800 sticky top-24 shadow-xl">
            <h3 className="text-lg font-bold text-slate-900 dark:text-white mb-6 border-b border-slate-100 dark:border-slate-800 pb-4">اطلاعات پروژه</h3>
            
            <div className="space-y-6">
              <div>
                <span className="flex items-center gap-2 text-slate-500 text-sm mb-1">
                  <User size={16} /> مشتری
                </span>
                <span className="text-slate-900 dark:text-white font-medium">{project.client}</span>
              </div>
              
              <div>
                <span className="flex items-center gap-2 text-slate-500 text-sm mb-1">
                  <Calendar size={16} /> سال اجرا
                </span>
                <span className="text-slate-900 dark:text-white font-medium">{project.year}</span>
              </div>

              <div>
                <span className="flex items-center gap-2 text-slate-500 text-sm mb-3">
                  <Tag size={16} /> تکنولوژی‌ها
                </span>
                <div className="flex flex-wrap gap-2">
                  {project.tags.map(tag => (
                    <span key={tag} className="px-3 py-1 bg-slate-100 dark:bg-slate-800 rounded-lg text-xs text-indigo-600 dark:text-indigo-300 border border-slate-200 dark:border-slate-700">
                      {tag}
                    </span>
                  ))}
                </div>
              </div>

              {project.projectUrl && (
                <a
                  href={project.projectUrl}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="w-full mt-4 bg-indigo-600 hover:bg-indigo-500 text-white font-bold py-3 px-4 rounded-xl flex items-center justify-center gap-2 transition-all shadow-lg shadow-indigo-600/20 hover:scale-[1.02]"
                >
                مشاهده آنلاین
                <ExternalLink size={18} />
                </a>
              )}
            </div>
          </div>
        </div>
      </div>

      {/* Related Projects */}
      {relatedProjects.length > 0 && (
          <div className="container mx-auto px-4 mt-20">
              <h2 className="text-2xl font-bold text-slate-900 dark:text-white mb-8 pr-2 border-r-4 border-indigo-600">پروژه‌های مرتبط</h2>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  {relatedProjects.map(rel => (
                      <a 
                      href={`#project/${rel.id}`}
                      onClick={(e) => handleNavClick(e, `#project/${rel.id}`)}
                      key={rel.id} 
                      className="group relative rounded-2xl overflow-hidden bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 hover:shadow-xl transition-all"
                      >
                          <div className="h-48 overflow-hidden">
                              <img src={rel.image} alt={rel.title} className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500" />
                          </div>
                          <div className="p-4">
                              <h3 className="font-bold text-slate-900 dark:text-white group-hover:text-indigo-500 transition-colors">{rel.title}</h3>
                              <p className="text-xs text-slate-500 mt-1">{rel.category}</p>
                          </div>
                      </a>
                  ))}
              </div>
          </div>
      )}

    </div>
  );
};

export default ProjectDetail;