import React, { useState, useEffect } from 'react';
import { Search, Save, Globe, Share2, FileText, BarChart3, Loader2 } from 'lucide-react';
import { AppSettings, SiteSEO } from '../types';
import { DataService, fireToast } from '../services/dataService';

interface SEOManagerProps {
  settings: AppSettings | null;
  onUpdate: () => void;
}

const SEOManager: React.FC<SEOManagerProps> = ({ settings, onUpdate }) => {
  const [localSEO, setLocalSEO] = useState<SiteSEO | null>(null);
  const [activeTab, setActiveTab] = useState<'basic' | 'og' | 'twitter' | 'schema' | 'analytics'>('basic');
  const [saving, setSaving] = useState(false);

  useEffect(() => {
    if (settings?.seo) {
      setLocalSEO(settings.seo);
    } else {
      setLocalSEO({
        title: '',
        description: '',
        keywords: [],
        author: '',
        canonicalUrl: '',
        robots: 'index, follow',
        ogTitle: '',
        ogDescription: '',
        ogImage: '',
        ogUrl: '',
        ogType: 'website',
        ogSiteName: '',
        ogLocale: 'fa_IR',
        twitterCard: 'summary_large_image',
        twitterSite: '',
        twitterCreator: '',
        twitterTitle: '',
        twitterDescription: '',
        twitterImage: '',
        schemaType: 'Organization',
        schemaName: '',
        schemaDescription: '',
        schemaLogo: '',
        schemaUrl: '',
        schemaEmail: '',
        schemaPhone: '',
        schemaAddress: {},
        schemaSameAs: [],
        googleAnalyticsId: '',
        googleSearchConsole: '',
        bingVerification: '',
        yandexVerification: '',
        favicon: '',
        appleTouchIcon: '',
        themeColor: ''
      });
    }
  }, [settings]);

  if (!localSEO) return null;

  const handleSave = async () => {
    if (!settings) return;
    
    setSaving(true);
    try {
      const updatedSettings: AppSettings = {
        ...settings,
        seo: localSEO
      };
      
      await DataService.updateSettings(updatedSettings);
      fireToast('تنظیمات SEO با موفقیت ذخیره شد', 'success');
      onUpdate();
    } catch (error) {
      fireToast('خطا در ذخیره تنظیمات SEO', 'error');
      console.error('SEO save error:', error);
    } finally {
      setSaving(false);
    }
  };

  const updateSEO = (field: keyof SiteSEO, value: any) => {
    setLocalSEO(prev => prev ? { ...prev, [field]: value } : null);
  };

  const tabs = [
    { id: 'basic' as const, label: 'متا تگ‌های پایه', icon: FileText },
    { id: 'og' as const, label: 'Open Graph', icon: Share2 },
    { id: 'twitter' as const, label: 'Twitter Cards', icon: Share2 },
    { id: 'schema' as const, label: 'Schema.org', icon: Globe },
    { id: 'analytics' as const, label: 'Analytics & Verification', icon: BarChart3 }
  ];

  return (
    <div className="space-y-6 animate-fade-in">
      <div className="flex justify-between items-center">
        <h2 className="text-2xl font-bold dark:text-white flex items-center gap-2">
          <Search className="text-indigo-500" size={24} />
          مدیریت SEO
        </h2>
        <button
          onClick={handleSave}
          disabled={saving}
          className="btn-primary flex items-center gap-2 disabled:opacity-50"
        >
          <Save size={18} />
          {saving ? 'در حال ذخیره...' : 'ذخیره تنظیمات SEO'}
        </button>
      </div>

      {/* Tabs */}
      <div className="flex gap-2 overflow-x-auto pb-2 border-b border-slate-200 dark:border-slate-700">
        {tabs.map(tab => {
          const Icon = tab.icon;
          return (
            <button
              key={tab.id}
              onClick={() => setActiveTab(tab.id)}
              className={`flex items-center gap-2 px-4 py-2 rounded-lg font-medium whitespace-nowrap transition-colors ${
                activeTab === tab.id
                  ? 'bg-indigo-600 text-white'
                  : 'bg-slate-100 dark:bg-slate-800 text-slate-600 dark:text-slate-400 hover:bg-slate-200 dark:hover:bg-slate-700'
              }`}
            >
              <Icon size={18} />
              {tab.label}
            </button>
          );
        })}
      </div>

      {/* Basic Meta Tags */}
      {activeTab === 'basic' && (
        <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl border border-slate-200 dark:border-slate-700 space-y-6">
          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">
              عنوان صفحه (Title) <span className="text-red-500">*</span>
              <span className="text-xs text-slate-500 block mt-1">
                {localSEO.title?.length || 0}/60 کاراکتر (بهینه: 50-60)
              </span>
            </label>
            <input
              type="text"
              className="input-field"
              placeholder="مثال: ویراوینگ - طراحی سایت و اپلیکیشن"
              value={localSEO.title || ''}
              onChange={e => updateSEO('title', e.target.value)}
              maxLength={60}
            />
          </div>

          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">
              توضیحات (Meta Description) <span className="text-red-500">*</span>
              <span className="text-xs text-slate-500 block mt-1">
                {localSEO.description?.length || 0}/160 کاراکتر (بهینه: 150-160)
              </span>
            </label>
            <textarea
              className="input-field"
              rows={3}
              placeholder="توضیحات کوتاه و جذاب برای موتورهای جستجو..."
              value={localSEO.description || ''}
              onChange={e => updateSEO('description', e.target.value)}
              maxLength={160}
            />
          </div>

          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">
              کلمات کلیدی (Keywords)
            </label>
            <input
              type="text"
              className="input-field"
              placeholder="مثال: طراحی سایت، توسعه اپلیکیشن، سئو"
              value={Array.isArray(localSEO.keywords) ? localSEO.keywords.join(', ') : (localSEO.keywords || '')}
              onChange={e => {
                const keywords = e.target.value.split(',').map(k => k.trim()).filter(k => k);
                updateSEO('keywords', keywords);
              }}
            />
            <p className="text-xs text-slate-500 mt-1">کلمات را با کاما جدا کنید</p>
          </div>

          <div className="grid md:grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium mb-2 dark:text-white">نویسنده (Author)</label>
              <input
                type="text"
                className="input-field"
                placeholder="نام نویسنده یا شرکت"
                value={localSEO.author || ''}
                onChange={e => updateSEO('author', e.target.value)}
              />
            </div>

            <div>
              <label className="block text-sm font-medium mb-2 dark:text-white">Canonical URL</label>
              <input
                type="text"
                className="input-field dir-ltr"
                placeholder="https://example.com"
                value={localSEO.canonicalUrl || ''}
                onChange={e => updateSEO('canonicalUrl', e.target.value)}
              />
            </div>
          </div>

          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">Robots Meta</label>
            <select
              className="input-field"
              value={localSEO.robots || 'index, follow'}
              onChange={e => updateSEO('robots', e.target.value)}
            >
              <option value="index, follow">Index, Follow</option>
              <option value="index, nofollow">Index, NoFollow</option>
              <option value="noindex, follow">NoIndex, Follow</option>
              <option value="noindex, nofollow">NoIndex, NoFollow</option>
            </select>
          </div>

          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">Theme Color</label>
            <input
              type="color"
              className="w-full h-12 rounded-lg border border-slate-200 dark:border-slate-700"
              value={localSEO.themeColor || '#6366f1'}
              onChange={e => updateSEO('themeColor', e.target.value)}
            />
          </div>
        </div>
      )}

      {/* Open Graph */}
      {activeTab === 'og' && (
        <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl border border-slate-200 dark:border-slate-700 space-y-6">
          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">OG Title</label>
            <input
              type="text"
              className="input-field"
              placeholder="عنوان برای اشتراک‌گذاری در Facebook/LinkedIn"
              value={localSEO.ogTitle || ''}
              onChange={e => updateSEO('ogTitle', e.target.value)}
            />
          </div>

          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">OG Description</label>
            <textarea
              className="input-field"
              rows={3}
              placeholder="توضیحات برای اشتراک‌گذاری..."
              value={localSEO.ogDescription || ''}
              onChange={e => updateSEO('ogDescription', e.target.value)}
            />
          </div>

          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">OG Image URL</label>
            <input
              type="text"
              className="input-field dir-ltr"
              placeholder="https://example.com/image.jpg (پیشنهادی: 1200x630)"
              value={localSEO.ogImage || ''}
              onChange={e => updateSEO('ogImage', e.target.value)}
            />
          </div>

          <div className="grid md:grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium mb-2 dark:text-white">OG URL</label>
              <input
                type="text"
                className="input-field dir-ltr"
                placeholder="https://example.com"
                value={localSEO.ogUrl || ''}
                onChange={e => updateSEO('ogUrl', e.target.value)}
              />
            </div>

            <div>
              <label className="block text-sm font-medium mb-2 dark:text-white">OG Type</label>
              <select
                className="input-field"
                value={localSEO.ogType || 'website'}
                onChange={e => updateSEO('ogType', e.target.value)}
              >
                <option value="website">Website</option>
                <option value="article">Article</option>
                <option value="business.business">Business</option>
              </select>
            </div>
          </div>

          <div className="grid md:grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium mb-2 dark:text-white">OG Site Name</label>
              <input
                type="text"
                className="input-field"
                placeholder="نام سایت"
                value={localSEO.ogSiteName || settings?.siteName || ''}
                onChange={e => updateSEO('ogSiteName', e.target.value)}
              />
            </div>

            <div>
              <label className="block text-sm font-medium mb-2 dark:text-white">OG Locale</label>
              <input
                type="text"
                className="input-field"
                placeholder="fa_IR"
                value={localSEO.ogLocale || 'fa_IR'}
                onChange={e => updateSEO('ogLocale', e.target.value)}
              />
            </div>
          </div>
        </div>
      )}

      {/* Twitter Cards */}
      {activeTab === 'twitter' && (
        <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl border border-slate-200 dark:border-slate-700 space-y-6">
          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">Twitter Card Type</label>
            <select
              className="input-field"
              value={localSEO.twitterCard || 'summary_large_image'}
              onChange={e => updateSEO('twitterCard', e.target.value)}
            >
              <option value="summary">Summary</option>
              <option value="summary_large_image">Summary Large Image</option>
            </select>
          </div>

          <div className="grid md:grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium mb-2 dark:text-white">Twitter Site (@username)</label>
              <input
                type="text"
                className="input-field dir-ltr"
                placeholder="@yourusername"
                value={localSEO.twitterSite || ''}
                onChange={e => updateSEO('twitterSite', e.target.value)}
              />
            </div>

            <div>
              <label className="block text-sm font-medium mb-2 dark:text-white">Twitter Creator (@username)</label>
              <input
                type="text"
                className="input-field dir-ltr"
                placeholder="@yourusername"
                value={localSEO.twitterCreator || ''}
                onChange={e => updateSEO('twitterCreator', e.target.value)}
              />
            </div>
          </div>

          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">Twitter Title</label>
            <input
              type="text"
              className="input-field"
              placeholder="عنوان برای Twitter"
              value={localSEO.twitterTitle || ''}
              onChange={e => updateSEO('twitterTitle', e.target.value)}
            />
          </div>

          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">Twitter Description</label>
            <textarea
              className="input-field"
              rows={3}
              placeholder="توضیحات برای Twitter..."
              value={localSEO.twitterDescription || ''}
              onChange={e => updateSEO('twitterDescription', e.target.value)}
            />
          </div>

          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">Twitter Image URL</label>
            <input
              type="text"
              className="input-field dir-ltr"
              placeholder="https://example.com/image.jpg (پیشنهادی: 1200x600)"
              value={localSEO.twitterImage || ''}
              onChange={e => updateSEO('twitterImage', e.target.value)}
            />
          </div>
        </div>
      )}

      {/* Schema.org */}
      {activeTab === 'schema' && (
        <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl border border-slate-200 dark:border-slate-700 space-y-6">
          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">Schema Type</label>
            <select
              className="input-field"
              value={localSEO.schemaType || 'Organization'}
              onChange={e => updateSEO('schemaType', e.target.value)}
            >
              <option value="Organization">Organization</option>
              <option value="LocalBusiness">Local Business</option>
              <option value="ProfessionalService">Professional Service</option>
              <option value="WebSite">Web Site</option>
            </select>
          </div>

          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">Schema Name</label>
            <input
              type="text"
              className="input-field"
              placeholder="نام سازمان یا کسب‌وکار"
              value={localSEO.schemaName || ''}
              onChange={e => updateSEO('schemaName', e.target.value)}
            />
          </div>

          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">Schema Description</label>
            <textarea
              className="input-field"
              rows={3}
              placeholder="توضیحات سازمان..."
              value={localSEO.schemaDescription || ''}
              onChange={e => updateSEO('schemaDescription', e.target.value)}
            />
          </div>

          <div className="grid md:grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium mb-2 dark:text-white">Schema Logo URL</label>
              <input
                type="text"
                className="input-field dir-ltr"
                placeholder="https://example.com/logo.png"
                value={localSEO.schemaLogo || ''}
                onChange={e => updateSEO('schemaLogo', e.target.value)}
              />
            </div>

            <div>
              <label className="block text-sm font-medium mb-2 dark:text-white">Schema URL</label>
              <input
                type="text"
                className="input-field dir-ltr"
                placeholder="https://example.com"
                value={localSEO.schemaUrl || ''}
                onChange={e => updateSEO('schemaUrl', e.target.value)}
              />
            </div>
          </div>

          <div className="grid md:grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium mb-2 dark:text-white">Schema Email</label>
              <input
                type="email"
                className="input-field dir-ltr"
                placeholder="info@example.com"
                value={localSEO.schemaEmail || ''}
                onChange={e => updateSEO('schemaEmail', e.target.value)}
              />
            </div>

            <div>
              <label className="block text-sm font-medium mb-2 dark:text-white">Schema Phone</label>
              <input
                type="text"
                className="input-field"
                placeholder="021-12345678"
                value={localSEO.schemaPhone || ''}
                onChange={e => updateSEO('schemaPhone', e.target.value)}
              />
            </div>
          </div>

          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">Schema Same As (Social Media URLs)</label>
            <textarea
              className="input-field"
              rows={4}
              placeholder="هر URL در یک خط&#10;https://facebook.com/yourpage&#10;https://instagram.com/yourpage&#10;https://linkedin.com/company/yourcompany"
              value={Array.isArray(localSEO.schemaSameAs) ? localSEO.schemaSameAs.join('\n') : ''}
              onChange={e => {
                const urls = e.target.value.split('\n').map(u => u.trim()).filter(u => u);
                updateSEO('schemaSameAs', urls);
              }}
            />
            <p className="text-xs text-slate-500 mt-1">هر URL را در یک خط جداگانه وارد کنید</p>
          </div>
        </div>
      )}

      {/* Analytics & Verification */}
      {activeTab === 'analytics' && (
        <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl border border-slate-200 dark:border-slate-700 space-y-6">
          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">Google Analytics ID</label>
            <input
              type="text"
              className="input-field dir-ltr"
              placeholder="G-XXXXXXXXXX یا UA-XXXXXXXXX-X"
              value={localSEO.googleAnalyticsId || ''}
              onChange={e => updateSEO('googleAnalyticsId', e.target.value)}
            />
            <p className="text-xs text-slate-500 mt-1">پشتیبانی از GA4 (G-...) و Universal Analytics (UA-...)</p>
          </div>

          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">Google Search Console Verification</label>
            <input
              type="text"
              className="input-field dir-ltr"
              placeholder="کد تأیید Google Search Console"
              value={localSEO.googleSearchConsole || ''}
              onChange={e => updateSEO('googleSearchConsole', e.target.value)}
            />
          </div>

          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">Bing Verification</label>
            <input
              type="text"
              className="input-field dir-ltr"
              placeholder="کد تأیید Bing"
              value={localSEO.bingVerification || ''}
              onChange={e => updateSEO('bingVerification', e.target.value)}
            />
          </div>

          <div>
            <label className="block text-sm font-medium mb-2 dark:text-white">Yandex Verification</label>
            <input
              type="text"
              className="input-field dir-ltr"
              placeholder="کد تأیید Yandex"
              value={localSEO.yandexVerification || ''}
              onChange={e => updateSEO('yandexVerification', e.target.value)}
            />
          </div>
        </div>
      )}

      {/* Save Button (Sticky) */}
      <div className="sticky bottom-4 bg-white dark:bg-slate-900 p-4 rounded-xl border border-slate-200 dark:border-slate-700 shadow-lg">
        <button
          onClick={handleSave}
          disabled={saving}
          className="w-full py-3 bg-indigo-600 hover:bg-indigo-700 text-white font-bold rounded-xl transition-colors disabled:opacity-50 disabled:cursor-not-allowed flex items-center justify-center gap-2"
        >
          {saving ? (
            <>
              <Loader2 className="w-5 h-5 animate-spin" />
              در حال ذخیره...
            </>
          ) : (
            <>
              <Save size={20} />
              ذخیره تنظیمات SEO
            </>
          )}
        </button>
      </div>
    </div>
  );
};

export default SEOManager;

