import React, { useState, useEffect } from 'react';
import { Linkedin, Github, Twitter } from 'lucide-react';
import { DataService } from '../services/dataService';
import { TeamMember } from '../types';

const Team: React.FC = () => {
  const [team, setTeam] = useState<TeamMember[]>([]);

  useEffect(() => {
    const updateTeam = async () => {
      const data = await DataService.getTeam();
      setTeam(data);
    };
    updateTeam();
    window.addEventListener('data-updated', updateTeam);
    return () => window.removeEventListener('data-updated', updateTeam);
  }, []);

  return (
    <section className="py-16 md:py-24 bg-slate-50 dark:bg-slate-900 transition-colors duration-300">
      <div className="container mx-auto md:px-8 lg:px-12 xl:px-4">
        <div className="text-center mb-10 md:mb-16 px-4">
          <h2 className="text-2xl md:text-5xl font-bold text-slate-900 dark:text-white mb-4">متخصصان ویراوینگ</h2>
          <div className="h-1 w-24 bg-indigo-600 mx-auto rounded-full"></div>
          <p className="text-slate-600 dark:text-slate-400 mt-4 text-sm md:text-base">تیمی که رویاهای شما را می‌سازد</p>
        </div>

        {/* Mobile: Horizontal Scroll | Desktop: Grid */}
        <div className="flex overflow-x-auto gap-4 px-4 pb-8 snap-x snap-mandatory hide-scrollbar md:grid md:grid-cols-2 xl:grid-cols-4 md:gap-8 lg:gap-10 xl:gap-12 md:pb-0">
          {team.map((member) => (
            <div 
              key={member.id} 
              className="min-w-[260px] w-[70vw] md:w-auto snap-center flex-shrink-0 group bg-white dark:bg-slate-800/50 rounded-3xl p-6 border border-slate-200 dark:border-slate-700 hover:bg-white dark:hover:bg-slate-800 transition-all duration-300 md:hover:-translate-y-2 shadow-sm hover:shadow-xl relative overflow-hidden flex flex-col items-center"
            >
              {/* Mobile Background Decoration */}
              <div className="absolute top-0 left-0 w-full h-20 bg-gradient-to-b from-indigo-50 to-transparent dark:from-indigo-900/20 dark:to-transparent opacity-50"></div>

              <div className="relative mb-4 mx-auto w-24 h-24 md:w-32 md:h-32">
                <div className="absolute inset-0 bg-indigo-600 rounded-full blur-lg opacity-20 group-hover:opacity-50 transition-opacity"></div>
                <img 
                  src={member.image} 
                  alt={member.name} 
                  className="w-full h-full rounded-full object-cover border-4 border-white dark:border-slate-900 relative z-10 shadow-lg" 
                />
              </div>
              
              <div className="text-center relative z-10 flex-1 flex flex-col items-center w-full">
                <h3 className="text-lg md:text-xl font-bold text-slate-900 dark:text-white mb-2">{member.name}</h3>
                <div className="mb-4">
                   <span className="inline-block px-3 py-1 rounded-full bg-indigo-50 dark:bg-indigo-900/30 text-indigo-600 dark:text-indigo-400 text-xs font-bold border border-indigo-100 dark:border-indigo-500/20">
                    {member.role}
                   </span>
                </div>
                <p className="text-slate-600 dark:text-slate-400 text-xs md:text-sm mb-6 leading-relaxed flex-1 w-full text-center">
                  {member.bio}
                </p>
                <div className="flex justify-center gap-4 pt-4 border-t border-slate-100 dark:border-slate-700/50 w-full">
                  <a href={member.socials?.linkedin || '#'} className="p-2 rounded-full bg-slate-100 dark:bg-slate-700/50 text-slate-500 hover:text-indigo-600 dark:text-slate-400 dark:hover:text-white transition-colors"><Linkedin size={16} /></a>
                  <a href={member.socials?.github || '#'} className="p-2 rounded-full bg-slate-100 dark:bg-slate-700/50 text-slate-500 hover:text-indigo-600 dark:text-slate-400 dark:hover:text-white transition-colors"><Github size={16} /></a>
                  <a href={member.socials?.twitter || '#'} className="p-2 rounded-full bg-slate-100 dark:bg-slate-700/50 text-slate-500 hover:text-indigo-600 dark:text-slate-400 dark:hover:text-white transition-colors"><Twitter size={16} /></a>
                </div>
              </div>
            </div>
          ))}
          {/* Spacer */}
          <div className="w-1 flex-shrink-0 md:hidden"></div>
        </div>
      </div>
    </section>
  );
};

export default Team;