import { Project, BlogPost, ContactMessage, AppSettings, TeamMember, Review, AdminUser, DBConfig, Service, ProcessStep, PaginatedResponse, UploadResponse, MediaItem } from '../types';

// --- CONFIGURATION ---
// Use relative path '/api' so it works both in dev (via Vite proxy) and prod (Liara)
const API_BASE_URL = '/api';

const STORAGE_KEYS = {
  SERVICES: 'viraving_services',
  PROJECTS: 'viraving_projects',
  BLOG: 'viraving_blog',
  MESSAGES: 'viraving_messages',
  TEAM: 'viraving_team',
  REVIEWS: 'viraving_reviews',
  SETTINGS: 'viraving_settings',
  INSTALLED: 'viraving_installed',
  DB_CONFIG: 'viraving_db_config',
  ADMIN_USER: 'viraving_admin_user',
  AUTH_TOKEN: 'viraving_auth_token',
  PROCESS: 'viraving_process',
  USE_API: 'viraving_use_real_api' // New Key
};

// Check preference (Default to TRUE if we are in production build, otherwise check storage)
const useRealApi = () => {
    // In production (Liara), always use real API
    if (import.meta.env.PROD || import.meta.env.MODE === 'production') return true;
    // In development, check localStorage preference
    return localStorage.getItem(STORAGE_KEYS.USE_API) === 'true';
};

// --- HELPER FOR TOAST NOTIFICATIONS ---
export const fireToast = (message: string, type: 'success' | 'error' | 'info' = 'info') => {
  const event = new CustomEvent('viraving-toast', { detail: { message, type } });
  window.dispatchEvent(event);
};

// --- MOCK DATA ---
const initialServices: Service[] = [
  { id: '1', title: 'طراحی وب‌سایت', description: 'طراحی وب‌سایت‌های اختصاصی، شرکتی و فروشگاهی با رابط کاربری جذاب.', iconName: 'Globe' },
  { id: '2', title: 'توسعه اپلیکیشن', description: 'ساخت اپلیکیشن‌های موبایل (Android & iOS) با پرفورمنس بالا.', iconName: 'Smartphone' },
  { id: '3', title: 'طراحی UI/UX', description: 'طراحی رابط و تجربه کاربری بر اساس روانشناسی کاربر.', iconName: 'Palette' },
  { id: '4', title: 'سئو و بهینه‌سازی', description: 'بهینه‌سازی سایت برای موتورهای جستجو جهت افزایش ترافیک.', iconName: 'Search' },
  { id: '5', title: 'توسعه بک‌اند', description: 'پیاده‌سازی سیستم‌های قدرتمند سمت سرور و APIها.', iconName: 'Server' },
  { id: '6', title: 'امنیت سایبری', description: 'تست نفوذ و ایمن‌سازی وب‌سایت‌ها و اپلیکیشن‌ها.', iconName: 'ShieldCheck' },
];

const initialProcessSteps: ProcessStep[] = [
  { id: '1', title: 'نیازسنجی و مشاوره', description: 'در جلسات مشاوره، اهداف کسب‌وکار شما را تحلیل کرده و بهترین نقشه راه را ترسیم می‌کنیم.', iconName: 'MessageSquare' },
  { id: '2', title: 'طراحی بصری (UI/UX)', description: 'تیم طراحی ما با خلق رابط کاربری جذاب و استاندارد، تجربه کاربر را اولویت قرار می‌دهد.', iconName: 'PenTool' },
  { id: '3', title: 'توسعه تخصصی', description: 'برنامه‌نویسان ما با جدیدترین تکنولوژی‌ها، ایده شما را به محصولی قدرتمند تبدیل می‌کنند.', iconName: 'Code2' },
  { id: '4', title: 'تست و راه‌اندازی', description: 'پس از تست‌های دقیق عملکرد و امنیت، پروژه شما با اطمینان کامل لانچ می‌شود.', iconName: 'Rocket' },
];

const initialProjects: Project[] = [
  {
    id: '1',
    title: 'فروشگاه دیجی‌شاپ',
    category: 'E-Commerce',
    image: 'https://picsum.photos/800/600?random=1',
    description: 'یک پلتفرم فروشگاهی جامع با قابلیت مدیریت انبار، درگاه پرداخت اختصاصی و پنل فروشندگان.',
    tags: ['Next.js', 'Node.js', 'PostgreSQL', 'Redis'],
    client: 'هلدینگ دیجی‌شاپ',
    year: '۱۴۰۳',
    seo: { metaTitle: 'فروشگاه اینترنتی دیجی‌شاپ', metaDescription: 'خرید آنلاین', keywords: ['فروشگاه', 'موبایل'], canonicalUrl: 'https://digishop.ir' },
    challenges: [
      { id: 'c1', title: 'مدیریت ترافیک بالا', description: 'استفاده از Redis برای کشینگ و کاهش بار دیتابیس در زمان کمپین‌ها.' },
      { id: 'c2', title: 'درگاه پرداخت اختصاصی', description: 'پیاده‌سازی سوئیچینگ هوشمند بین درگاه‌های مختلف بانکی برای پایداری ۱۰۰٪.' }
    ]
  },
  {
    id: '2',
    title: 'اپلیکیشن سلامت',
    category: 'Mobile App',
    image: 'https://picsum.photos/800/600?random=2',
    description: 'اپلیکیشن پایش سلامت با قابلیت اتصال به ساعت‌های هوشمند.',
    tags: ['React Native', 'TypeScript', 'HealthKit'],
    client: 'شرکت سلامت پارس',
    year: '۱۴۰۲',
    seo: { metaTitle: 'اپلیکیشن سلامت', metaDescription: 'پایش سلامت', keywords: ['سلامت', 'ورزش'] },
    challenges: [
      { id: 'c1', title: 'سینک بلادرنگ', description: 'همگام‌سازی داده‌های ضربان قلب با سرور بدون مصرف زیاد باتری.' }
    ]
  },
  { id: '3', title: 'داشبورد مدیریتی آلفا', category: 'Web App', image: 'https://picsum.photos/800/600?random=3', description: 'پنل ادمین سازمانی.', tags: ['Vue.js', 'Laravel'], client: 'گروه صنعتی آلفا', year: '۱۴۰۳', challenges: [] },
  { id: '4', title: 'سایت شرکتی نکست', category: 'Corporate', image: 'https://picsum.photos/800/600?random=4', description: 'وب‌سایت معرفی شرکت.', tags: ['React', 'GSAP'], client: 'معماری نکست', year: '۱۴۰۱', challenges: [] },
  { id: '5', title: 'سامانه آموزش آنلاین', category: 'Education', image: 'https://picsum.photos/800/600?random=5', description: 'پلتفرم کلاس آنلاین.', tags: ['Next.js', 'WebRTC'], client: 'آکادمی رشد', year: '۱۴۰۳', challenges: [] },
  { id: '6', title: 'کیف پول کریپتو', category: 'Fintech', image: 'https://picsum.photos/800/600?random=6', description: 'اپلیکیشن کیف پول.', tags: ['React Native', 'Solidity'], client: 'کریپتو لند', year: '۱۴۰۲', challenges: [] },
  { id: '7', title: 'سیستم انبارداری هوشمند', category: 'Web App', image: 'https://picsum.photos/800/600?random=7', description: 'مدیریت موجودی انبار.', tags: ['React', 'Node.js'], client: 'پخش عقاب', year: '۱۴۰۳', challenges: [] },
  { id: '8', title: 'سایت گردشگری سفر', category: 'Tourism', image: 'https://picsum.photos/800/600?random=8', description: 'رزرو تور و هتل.', tags: ['Next.js', 'Strapi'], client: 'آژانس سفر', year: '۱۴۰۲', challenges: [] },
];

const initialBlogPosts: BlogPost[] = [
  { id: '1', title: 'تاثیر هوش مصنوعی بر آینده طراحی وب', excerpt: 'چگونه ابزارهای AI مانند Gemini فرآیند توسعه را متحول می‌کنند؟', content: '<p>محتوای کامل مقاله...</p>', date: '۲۰ مهر ۱۴۰۳', readTime: '۵ دقیقه', image: 'https://picsum.photos/600/400?random=10', author: 'آرش محمدی' },
  { id: '2', title: 'Next.js 14 و ویژگی‌های جدید', excerpt: 'بررسی Server Actions.', content: '<p>محتوای کامل...</p>', date: '۱۵ مهر ۱۴۰۳', readTime: '۸ دقیقه', image: 'https://picsum.photos/600/400?random=11', author: 'پویا کریمی' },
  { id: '3', title: 'اصول روانشناسی در UX', excerpt: 'قوانین گشتالت.', content: '<p>محتوای کامل...</p>', date: '۱۰ مهر ۱۴۰۳', readTime: '۶ دقیقه', image: 'https://picsum.photos/600/400?random=12', author: 'سارا احمدی' },
  { id: '4', title: 'راهنمای امنیت در React', excerpt: 'جلوگیری از حملات XSS.', content: '<p>محتوای کامل...</p>', date: '۵ مهر ۱۴۰۳', readTime: '۷ دقیقه', image: 'https://picsum.photos/600/400?random=13', author: 'آرش محمدی' },
  { id: '5', title: 'بهینه‌سازی تصاویر وب', excerpt: 'فرمت‌های WebP و AVIF.', content: '<p>محتوای کامل...</p>', date: '۱ مهر ۱۴۰۳', readTime: '۴ دقیقه', image: 'https://picsum.photos/600/400?random=14', author: 'مینا رستمی' },
];

const initialTeam: TeamMember[] = [
  { id: '1', name: 'آرش محمدی', role: 'CTO', image: 'https://i.pravatar.cc/300?u=arash', bio: '۱۰ سال تجربه توسعه نرم‌افزار' },
  { id: '2', name: 'سارا احمدی', role: 'Senior Designer', image: 'https://i.pravatar.cc/300?u=sara', bio: 'متخصص UI/UX' },
  { id: '3', name: 'پویا کریمی', role: 'Frontend Dev', image: 'https://i.pravatar.cc/300?u=pouya', bio: 'توسعه‌دهنده React' },
  { id: '4', name: 'مینا رستمی', role: 'SEO Specialist', image: 'https://i.pravatar.cc/300?u=mina', bio: 'متخصص دیجیتال مارکتینگ' }
];

const initialReviews: Review[] = [
  { id: '1', name: 'سارا جلالی', role: 'مدیر بازاریابی', text: 'تیم حرفه‌ای و کاربلد.', avatar: 'https://i.pravatar.cc/150?u=1' },
  { id: '2', name: 'مهدی رضایی', role: 'CEO', text: 'کیفیت عالی.', avatar: 'https://i.pravatar.cc/150?u=2' },
  { id: '3', name: 'رضا کمالی', role: 'Founder', text: 'پشتیبانی خوب.', avatar: 'https://i.pravatar.cc/150?u=3' },
];

const initialSettings: AppSettings = {
  siteName: 'ویراوینگ',
  logoUrl: undefined,
  seo: { 
    title: 'ویراوینگ - طراحی سایت و اپلیکیشن', 
    description: 'ویراوینگ پیشرو در طراحی سایت، اپلیکیشن و راهکارهای هوشمند تحت وب. تیم حرفه‌ای ما آماده ارائه خدمات با کیفیت است.', 
    keywords: ['طراحی سایت', 'توسعه اپلیکیشن', 'سئو', 'طراحی UI/UX'],
    robots: 'index, follow',
    ogType: 'website',
    ogLocale: 'fa_IR',
    twitterCard: 'summary_large_image',
    schemaType: 'Organization'
  },
  hero: { badge: 'آینده وب اینجاست', title: 'ما رویاهای دیجیتال شما را', highlightedWord: 'خلق می‌کنیم', description: 'توسعه تخصصی وب و موبایل', ctaText: 'مشاوره رایگان' },
  footer: { socials: {}, contact: { address: 'تهران', phone: '021-88881234', email: 'info@viraving.com' } },
  navigation: {
    header: [
      { id: '1', name: 'خانه', href: '#home' },
      { id: '2', name: 'پروژه‌ها', href: '#portfolio' },
      { id: '3', name: 'وبلاگ', href: '#blog' },
      { id: '4', name: 'مشاوره هوشمند', href: '#home', anchor: 'ai-consultant' },
      { id: '5', name: 'تماس با ما', href: '#contact', special: true },
    ],
    footer: {
      quickAccess: [
        { id: 'f1', name: 'خانه', href: '#home' },
        { id: 'f2', name: 'نمونه کارها', href: '#portfolio' },
        { id: 'f3', name: 'مشاوره هوشمند', href: '#home', anchor: 'ai-consultant' },
        { id: 'f4', name: 'شروع همکاری', href: '#contact' },
      ],
      services: [
        { id: 's1', name: 'طراحی وب‌سایت', href: '#home', anchor: 'services' },
        { id: 's2', name: 'توسعه اپلیکیشن', href: '#home', anchor: 'services' },
        { id: 's3', name: 'طراحی UI/UX', href: '#home', anchor: 'services' },
        { id: 's4', name: 'سئو و مارکتینگ', href: '#home', anchor: 'services' },
      ]
    },
    mobile: [
      { id: 'm1', name: 'خانه', href: '#home', iconName: 'Home' },
      { id: 'm2', name: 'پروژه‌ها', href: '#portfolio', iconName: 'Briefcase' },
      { id: 'm3', name: 'مشاور AI', href: '#home', anchor: 'ai-consultant', iconName: 'Zap' },
      { id: 'm4', name: 'تماس', href: '#contact', iconName: 'Phone' },
    ]
  },
  aiConfig: { 
    enabled: true, 
    provider: 'liara',
    modelName: 'openai/gpt-5-mini:online',
    systemInstruction: 'شما یک مشاور فنی ارشد در شرکت نرم‌افزاری "ویراوینگ" هستید. وظیفه شما تحلیل ایده پروژه کاربر و پیشنهاد یک استک تکنولوژی مدرن و یک نقشه راه (Roadmap) کلی است. پاسخ باید کاملا به زبان فارسی باشد.',
    apiKey: ''
  },
  legal: {
    terms: '<h2>قوانین و مقررات</h2><p>استفاده از خدمات ویراوینگ به معنی پذیرش این قوانین است...</p>',
    privacy: '<h2>حریم خصوصی</h2><p>ما به حریم خصوصی شما احترام می‌گذاریم و اطلاعات شما نزد ما محفوظ است...</p>'
  }
};

const apiClient = async <T>(endpoint: string, method: string = 'GET', body?: any, isFormData: boolean = false): Promise<T> => {
  const token = localStorage.getItem(STORAGE_KEYS.AUTH_TOKEN);
  const headers: HeadersInit = {};
  if (token) headers['Authorization'] = `Bearer ${token}`;
  if (!isFormData) headers['Content-Type'] = 'application/json';

  try {
    const response = await fetch(`${API_BASE_URL}${endpoint}`, {
      method,
      headers,
      body: isFormData ? body : (body ? JSON.stringify(body) : undefined)
    });
    
    if (response.status === 401) {
      DataService.logout();
      throw new Error('Unauthorized');
    }
    if (!response.ok) throw new Error(`Error ${response.status}`);
    return await response.json();
  } catch (error) {
    console.error(`API Error:`, error);
    throw error;
  }
};

const updateStorage = (key: string, data: any) => {
  localStorage.setItem(key, JSON.stringify(data));
  window.dispatchEvent(new Event('data-updated'));
};

const getFromStorage = <T>(key: string, initial: T): T => {
  try {
    const saved = localStorage.getItem(key);
    return saved ? JSON.parse(saved) : initial;
  } catch {
    return initial;
  }
};

export const DataService = {
  isInstalled: async (): Promise<boolean> => {
      // In production, check via API
      if (import.meta.env.PROD || import.meta.env.MODE === 'production') {
        try {
          const response = await fetch(`${API_BASE_URL}/install/status`);
          if (response.ok) {
            const data = await response.json();
            return data.installed || false;
          }
          return false;
        } catch (error) {
          // If API is not available, assume not installed
          return false;
        }
      }
      return localStorage.getItem(STORAGE_KEYS.INSTALLED) === 'true';
  },
  
  isAuthenticated: (): boolean => !!localStorage.getItem(STORAGE_KEYS.AUTH_TOKEN),
  
  // API Toggle Methods
  getApiMode: (): boolean => useRealApi(),
  setApiMode: (enabled: boolean) => {
      localStorage.setItem(STORAGE_KEYS.USE_API, enabled ? 'true' : 'false');
      window.location.reload(); // Reload to apply changes fresh
  },
  
  // API Client (for use in other services)
  apiClient: apiClient,

  installApp: async (dbConfig: DBConfig, adminUser: AdminUser) => {
    if (useRealApi()) {
        await apiClient('/install', 'POST', { dbConfig, adminUser });
    }
    // Always set local flag so app loads
    localStorage.setItem(STORAGE_KEYS.DB_CONFIG, JSON.stringify(dbConfig));
    localStorage.setItem(STORAGE_KEYS.ADMIN_USER, JSON.stringify(adminUser));
    localStorage.setItem(STORAGE_KEYS.INSTALLED, 'true');
    
    // Seed Local Data just in case they switch back
    updateStorage(STORAGE_KEYS.SERVICES, initialServices);
    updateStorage(STORAGE_KEYS.PROJECTS, initialProjects);
    updateStorage(STORAGE_KEYS.SETTINGS, initialSettings);
    updateStorage(STORAGE_KEYS.PROCESS, initialProcessSteps);
    updateStorage(STORAGE_KEYS.BLOG, initialBlogPosts);
    updateStorage(STORAGE_KEYS.TEAM, initialTeam);
    updateStorage(STORAGE_KEYS.REVIEWS, initialReviews);
    
    window.dispatchEvent(new Event('app-installed'));
  },

  uploadFile: async (file: File): Promise<string> => {
    if (useRealApi()) {
      const formData = new FormData();
      formData.append('file', file);
      const res = await apiClient<UploadResponse>('/upload', 'POST', formData, true);
      return res.url;
    } else {
      return new Promise((resolve, reject) => {
        const reader = new FileReader();
        reader.onloadend = () => resolve(reader.result as string);
        reader.onerror = reject;
        reader.readAsDataURL(file);
      });
    }
  },

  // Media Library
  getMedia: async (): Promise<MediaItem[]> => {
    if (useRealApi()) {
      return apiClient<MediaItem[]>('/media');
    }
    // In local mode, media library is not tracked separately
    return [];
  },

  deleteMedia: async (filename: string): Promise<void> => {
    if (useRealApi()) {
      await apiClient(`/media/${encodeURIComponent(filename)}`, 'DELETE');
    }
  },

  getProjects: async (page: number = 1, limit: number = 100, search?: string, category?: string): Promise<{ data: Project[], total: number }> => {
    if (useRealApi()) {
       let query = `/projects?page=${page}&limit=${limit}`;
       if(search) query += `&search=${encodeURIComponent(search)}`;
       if(category && category !== 'All') query += `&category=${encodeURIComponent(category)}`;
       return apiClient(query);
    } else {
       let all = getFromStorage(STORAGE_KEYS.PROJECTS, initialProjects);
       
       if(search) {
         const lower = search.toLowerCase();
         all = all.filter(p => p.title.toLowerCase().includes(lower) || p.description.toLowerCase().includes(lower));
       }
       
       if (category && category !== 'All') {
         all = all.filter(p => p.category === category);
       }

       const start = (page - 1) * limit;
       return {
         data: all.slice(start, start + limit),
         total: all.length
       };
    }
  },

  getProjectById: async (id: string): Promise<Project | undefined> => {
    if (useRealApi()) {
       return apiClient(`/projects/${id}`);
    } else {
       const all = getFromStorage(STORAGE_KEYS.PROJECTS, initialProjects);
       return all.find(p => p.id === id);
    }
  },
  
  saveProject: async (project: Project) => {
    if (useRealApi()) {
      await apiClient('/projects', project.id ? 'PUT' : 'POST', project);
    } else {
      const items = getFromStorage(STORAGE_KEYS.PROJECTS, initialProjects);
      const index = items.findIndex(p => p.id === project.id);
      if (index >= 0) items[index] = project;
      else items.push({ ...project, id: Date.now().toString() });
      updateStorage(STORAGE_KEYS.PROJECTS, items);
    }
  },

  deleteProject: async (id: string) => {
    if (useRealApi()) {
      await apiClient(`/projects/${id}`, 'DELETE');
    } else {
      const items = getFromStorage(STORAGE_KEYS.PROJECTS, initialProjects).filter(p => p.id !== id);
      updateStorage(STORAGE_KEYS.PROJECTS, items);
    }
  },

  getServices: async (): Promise<Service[]> => {
    if (useRealApi()) {
        const res = await apiClient<{data: Service[]}>('/services');
        return res.data;
    }
    return getFromStorage(STORAGE_KEYS.SERVICES, initialServices);
  },

  saveService: async (service: Service) => {
    if (useRealApi()) await apiClient('/services', service.id ? 'PUT' : 'POST', service);
    else {
        const items = getFromStorage(STORAGE_KEYS.SERVICES, initialServices);
        const index = items.findIndex(s => s.id === service.id);
        if (index >= 0) items[index] = service;
        else items.push({ ...service, id: Date.now().toString() });
        updateStorage(STORAGE_KEYS.SERVICES, items);
    }
  },

  deleteService: async (id: string) => {
    if(useRealApi()) await apiClient(`/services/${id}`, 'DELETE');
    else {
        const items = getFromStorage(STORAGE_KEYS.SERVICES, initialServices).filter(s => s.id !== id);
        updateStorage(STORAGE_KEYS.SERVICES, items);
    }
  },

  getProcessSteps: async (): Promise<ProcessStep[]> => {
    if (useRealApi()) {
         const res = await apiClient<{data: ProcessStep[]}>('/process');
         return res.data;
    }
    return getFromStorage(STORAGE_KEYS.PROCESS, initialProcessSteps);
  },

  saveProcessStep: async (step: ProcessStep) => {
    if (useRealApi()) await apiClient('/process', step.id ? 'PUT' : 'POST', step);
    else {
        const items = getFromStorage(STORAGE_KEYS.PROCESS, initialProcessSteps);
        const index = items.findIndex(s => s.id === step.id);
        if (index >= 0) items[index] = step;
        else items.push({ ...step, id: Date.now().toString() });
        updateStorage(STORAGE_KEYS.PROCESS, items);
    }
  },

  deleteProcessStep: async (id: string) => {
    if(useRealApi()) await apiClient(`/process/${id}`, 'DELETE');
    else {
        const items = getFromStorage(STORAGE_KEYS.PROCESS, initialProcessSteps).filter(s => s.id !== id);
        updateStorage(STORAGE_KEYS.PROCESS, items);
    }
  },

  getBlogPosts: async (page: number = 1, limit: number = 100, search?: string): Promise<{ data: BlogPost[], total: number }> => {
    if (useRealApi()) {
      let query = `/blog?page=${page}&limit=${limit}`;
      if(search) query += `&search=${encodeURIComponent(search)}`;
      return apiClient(query);
    }
    let all = getFromStorage(STORAGE_KEYS.BLOG, initialBlogPosts);
    if(search) {
       const lower = search.toLowerCase();
       all = all.filter(p => p.title.toLowerCase().includes(lower) || p.excerpt.toLowerCase().includes(lower));
    }
    const start = (page - 1) * limit;
    return {
        data: all.slice(start, start + limit),
        total: all.length
    };
  },

  getPostById: async (id: string): Promise<BlogPost | undefined> => {
    if (useRealApi()) return apiClient(`/blog/${id}`);
    const posts = getFromStorage(STORAGE_KEYS.BLOG, initialBlogPosts);
    return posts.find(p => p.id === id);
  },

  saveBlogPost: async (post: BlogPost) => {
    if (useRealApi()) await apiClient('/blog', post.id ? 'PUT' : 'POST', post);
    else {
        const items = getFromStorage(STORAGE_KEYS.BLOG, initialBlogPosts);
        const index = items.findIndex(p => p.id === post.id);
        if (index >= 0) items[index] = post;
        else items.push({ ...post, id: Date.now().toString() });
        updateStorage(STORAGE_KEYS.BLOG, items);
    }
  },

  deleteBlogPost: async (id: string) => {
    if (useRealApi()) await apiClient(`/blog/${id}`, 'DELETE');
    else {
        const items = getFromStorage(STORAGE_KEYS.BLOG, initialBlogPosts).filter(p => p.id !== id);
        updateStorage(STORAGE_KEYS.BLOG, items);
    }
  },

  getTeam: async (): Promise<TeamMember[]> => {
    if (useRealApi()) {
        const res = await apiClient<{data: TeamMember[]}>('/team');
        return res.data;
    }
    return getFromStorage(STORAGE_KEYS.TEAM, initialTeam);
  },

  saveTeamMember: async (member: TeamMember) => {
    if (useRealApi()) await apiClient('/team', member.id ? 'PUT' : 'POST', member);
    else {
        const items = getFromStorage(STORAGE_KEYS.TEAM, initialTeam);
        const index = items.findIndex(t => t.id === member.id);
        if (index >= 0) items[index] = member;
        else items.push({ ...member, id: Date.now().toString() });
        updateStorage(STORAGE_KEYS.TEAM, items);
    }
  },

  deleteTeamMember: async (id: string) => {
    if (useRealApi()) await apiClient(`/team/${id}`, 'DELETE');
    else {
        const items = getFromStorage(STORAGE_KEYS.TEAM, initialTeam).filter(t => t.id !== id);
        updateStorage(STORAGE_KEYS.TEAM, items);
    }
  },

  getReviews: async (): Promise<Review[]> => {
    if (useRealApi()) {
        const res = await apiClient<{data: Review[]}>('/reviews');
        return res.data;
    }
    return getFromStorage(STORAGE_KEYS.REVIEWS, initialReviews);
  },

  saveReview: async (review: Review) => {
    if (useRealApi()) await apiClient('/reviews', review.id ? 'PUT' : 'POST', review);
    else {
        const items = getFromStorage(STORAGE_KEYS.REVIEWS, initialReviews);
        const index = items.findIndex(r => r.id === review.id);
        if (index >= 0) items[index] = review;
        else items.push({ ...review, id: Date.now().toString() });
        updateStorage(STORAGE_KEYS.REVIEWS, items);
    }
  },

  deleteReview: async (id: string) => {
    if (useRealApi()) await apiClient(`/reviews/${id}`, 'DELETE');
    else {
        const items = getFromStorage(STORAGE_KEYS.REVIEWS, initialReviews).filter(r => r.id !== id);
        updateStorage(STORAGE_KEYS.REVIEWS, items);
    }
  },

  getMessages: async (page: number = 1, limit: number = 100): Promise<{ data: ContactMessage[], total: number }> => {
    if (useRealApi()) return apiClient(`/messages?page=${page}&limit=${limit}`);
    const all = getFromStorage(STORAGE_KEYS.MESSAGES, []);
    const start = (page - 1) * limit;
    return {
        data: all.slice(start, start + limit),
        total: all.length
    };
  },

  addMessage: async (msg: Omit<ContactMessage, 'id' | 'date' | 'isRead'>) => {
    if (useRealApi()) await apiClient('/messages', 'POST', msg);
    else {
        const items = getFromStorage(STORAGE_KEYS.MESSAGES, []);
        const newMsg = { ...msg, id: Date.now().toString(), date: new Date().toLocaleDateString('fa-IR'), isRead: false };
        items.unshift(newMsg);
        updateStorage(STORAGE_KEYS.MESSAGES, items);
    }
  },

  markAsRead: async (id: string) => {
    if (useRealApi()) await apiClient(`/messages/${id}/read`, 'PUT');
    else {
        const items = getFromStorage(STORAGE_KEYS.MESSAGES, []).map((m: any) => m.id === id ? { ...m, isRead: true } : m);
        updateStorage(STORAGE_KEYS.MESSAGES, items);
    }
  },

  deleteMessage: async (id: string) => {
    if (useRealApi()) await apiClient(`/messages/${id}`, 'DELETE');
    else {
        const items = getFromStorage(STORAGE_KEYS.MESSAGES, []).filter((m: any) => m.id !== id);
        updateStorage(STORAGE_KEYS.MESSAGES, items);
    }
  },

  getDashboardStats: async (): Promise<{projects: number, blog: number, messages: number}> => {
    if (useRealApi()) return apiClient('/stats');
    return {
        projects: getFromStorage(STORAGE_KEYS.PROJECTS, initialProjects).length,
        blog: getFromStorage(STORAGE_KEYS.BLOG, initialBlogPosts).length,
        messages: getFromStorage(STORAGE_KEYS.MESSAGES, []).length
    };
  },

  getSettings: async (): Promise<AppSettings> => {
    if (useRealApi()) return apiClient('/settings');
    return getFromStorage(STORAGE_KEYS.SETTINGS, initialSettings);
  },

  updateSettings: async (settings: AppSettings) => {
    if (useRealApi()) {
        await apiClient('/settings', 'PUT', settings);
        window.dispatchEvent(new Event('settings-updated'));
    } else {
        updateStorage(STORAGE_KEYS.SETTINGS, settings);
        window.dispatchEvent(new Event('settings-updated'));
    }
  },

  login: async (username: string, pass: string): Promise<boolean> => {
    if (useRealApi()) {
        try {
            const response: { token: string } = await apiClient('/auth/login', 'POST', { username, password: pass });
            if (response && response.token) {
                localStorage.setItem(STORAGE_KEYS.AUTH_TOKEN, response.token);
                return true;
            }
            return false;
        } catch { return false; }
    } else {
        return new Promise(resolve => {
          setTimeout(() => {
            const admin = getFromStorage<AdminUser | null>(STORAGE_KEYS.ADMIN_USER, null);
            if (admin && admin.username === username && admin.password === pass) {
              localStorage.setItem(STORAGE_KEYS.AUTH_TOKEN, 'mock_valid_token');
              resolve(true);
            } else {
              resolve(false);
            }
          }, 500);
        });
    }
  },

  updateAdmin: async (userData: Partial<AdminUser>): Promise<boolean> => {
     if (useRealApi()) {
         await apiClient('/auth/profile', 'PUT', userData);
         return true;
     } else {
         const currentAdmin = getFromStorage<AdminUser | null>(STORAGE_KEYS.ADMIN_USER, null);
         if (currentAdmin) {
             const updated = { ...currentAdmin, ...userData };
             localStorage.setItem(STORAGE_KEYS.ADMIN_USER, JSON.stringify(updated));
             return true;
         }
         return false;
     }
  },

  logout: () => {
    localStorage.removeItem(STORAGE_KEYS.AUTH_TOKEN);
    window.location.hash = '#home';
    window.location.reload();
  }
};